<?php
namespace CollaborativeFilteringBase;

if ( !defined( 'COLLABORATIVE_FILTERING_PLUGIN' ) )
	exit;

abstract class CollaborativeFiltering_Base_Class
{

	private static $settings = array(

		"exclude_loggedin_user" => array( "label" => "whether to exclude logged in user", "type" => "bool", "default" => COLLABORATIVE_FILTERING_EXCLUDE_LOGGEDIN_USER, "url" => "https://technote.space/page-1456" ),
		"update_cookie_expire" => array( "label" => "whether to update cookie expiration when accessed or not", "type" => "bool", "default" => COLLABORATIVE_FILTERING_UPDATE_COOKIE_EXPIRE, "url" => "https://technote.space/page-1458" ),
		"user_expire" => array( "label" => "expiration of user's cookie", "type" => "int", "default" => COLLABORATIVE_FILTERING_USER_EXPIRE, "min" => 0, "url" => "https://technote.space/page-1462" ),
		"sampling_rate" => array( "label" => "sampling rate", "type" => "float", "default" => COLLABORATIVE_FILTERING_SAMPLING_RATE, "min" => 0, "max" => 1, "option" => "sampling_rate", "option_default" => COLLABORATIVE_FILTERING_DEFAULT_SAMPLING_RATE, "url" => "https://technote.space/page-1428" ),
		"check_data" => array( "label" => "whether to validate cookie data or not", "type" => "bool", "default" => COLLABORATIVE_FILTERING_CHECK_DATA, "url" => "https://technote.space/page-1435" ),
		"nonce_check" => array( "label" => "whether to check nonce or not", "type" => "bool", "default" => COLLABORATIVE_FILTERING_NONCE_CHECK, "url" => "https://technote.space/page-1438" ),
		"calc_interval" => array( "label" => "interval of calculate", "type" => "int", "default" => COLLABORATIVE_FILTERING_CALC_INTERVAL, "min" => MINUTE_IN_SECONDS, "url" => "https://technote.space/page-1466" ),
		"calc_timeout" => array( "label" => "timeout of calculate", "type" => "int", "default" => COLLABORATIVE_FILTERING_CALC_TIMEOUT, "min" => MINUTE_IN_SECONDS, "url" => "https://technote.space/page-1468" ),
		"calc_log" => array( "label" => "whether to log calculate process", "type" => "bool", "default" => COLLABORATIVE_FILTERING_CALC_LOG, "url" => "https://technote.space/page-1473" ),
		"clear_interval" => array( "label" => "interval of clear access data", "type" => "int", "default" => COLLABORATIVE_FILTERING_CLEAR_INTERVAL, "min" => MINUTE_IN_SECONDS, "url" => "https://technote.space/page-1475" ),
//		"clear_timeout" => array( "label" => "timeout of clear access data", "type" => "int", "default" => COLLABORATIVE_FILTERING_CLEAR_TIMEOUT, "min" => MINUTE_IN_SECONDS ),
		"clear_log" => array( "label" => "whether to log clear process", "type" => "bool", "default" => COLLABORATIVE_FILTERING_CLEAR_LOG, "url" => "https://technote.space/page-1479" ),
		"data_expire" => array( "label" => "expiration of access data", "type" => "int", "default" => COLLABORATIVE_FILTERING_DATA_EXPIRE, "min" => MINUTE_IN_SECONDS, "url" => "https://technote.space/page-1464" ),
		"display_log_number" => array( "label" => "number to display log", "type" => "int", "default" => COLLABORATIVE_FILTERING_DISPLAY_LOG_NUMBER, "min" => 0, "url" => "https://technote.space/page-1489" ),
		"max_save_data_number" => array( "label" => "max number to save calculated data", "type" => "int", "default" => COLLABORATIVE_FILTERING_MAX_SAVE_DATA_NUMBER, "min" => 1, "url" => "https://technote.space/page-1446" ),
		"minify_js" => array( "label" => "whether to minify js that generated by this plugin", "type" => "bool", "default" => COLLABORATIVE_FILTERING_MINIFY_JS, "url" => "https://technote.space/page-1430" ),
		"minify_css" => array( "label" => "whether to minify css that generated by this plugin", "type" => "bool", "default" => COLLABORATIVE_FILTERING_MINIFY_CSS, "url" => "https://technote.space/page-1432" ),
		"jaccard_threshold" => array( "label" => "threshold of jaccard to get results, which is used when argument is not supplied to function", "type" => "float", "default" => COLLABORATIVE_FILTERING_JACCARD_THRESHOLD, "min" => 0, "max" => 1, "url" => "https://technote.space/page-1448" ),
		"jaccard_min_number" => array( "label" => "min data number to be required to get results, which is used when argument is not supplied to function", "type" => "int", "default" => COLLABORATIVE_FILTERING_JACCARD_MIN_NUMBER, "min" => 0, "url" => "https://technote.space/page-1450" ),
		"calculate_number" => array( "label" => "max number of calculations of per", "type" => "int", "default" => COLLABORATIVE_FILTERING_CALCULATE_NUMBER, "min" => 10, "url" => "https://technote.space/page-1470" ),
		"post_types" => array( "label" => "valid post types, which is separated by commas", "type" => "string", "default" => COLLABORATIVE_FILTERING_POST_TYPES, "url" => "https://technote.space/page-1452" ),
		"post_statuses" => array( "label" => "valid post statuses, which is separated by commas", "type" => "string", "default" => COLLABORATIVE_FILTERING_POST_STATUSES, "url" => "https://technote.space/page-1454" ),
		"show_result" => array( "label" => "whether to set button to show result", "type" => "bool", "default" => COLLABORATIVE_FILTERING_SHOW_RESULT, "url" => "https://technote.space/page-1493" ),
		"front_admin_ajax" => array( "label" => "whether to use admin-ajax.php on front page", "type" => "bool", "default" => COLLABORATIVE_FILTERING_FRONT_ADMIN_AJAX, "url" => "https://technote.space/page-1442" ),
		"check_referer" => array( "label" => "whether to check referer when ajax access without nonce check", "type" => "bool", "default" => COLLABORATIVE_FILTERING_CHECK_REFERER, "url" => "https://technote.space/page-1439" ),
		"consider_page_cache" => array( "label" => "whether to consider page cache", "type" => "bool", "default" => COLLABORATIVE_FILTERING_CONSIDER_PAGE_CACHE, "url" => "https://technote.space/page-1481" ),
		"url_scheme" => array( "label" => "url scheme", "type" => "string", "default" => COLLABORATIVE_FILTERING_URL_SCHEME, "url" => "https://technote.space/page-1485" ),
		"utilize_wpp" => array( "label" => "whether to utilize wpp", "type" => "bool", "default" => COLLABORATIVE_FILTERING_UTILIZE_WPP, "url" => "https://technote.space/page-1434" ),
		"mod_rewrite_ajax_access" => array( "label" => "whether to use mod rewrite access when ajax access", "type" => "bool", "default" => COLLABORATIVE_FILTERING_MOD_REWRITE_AJAX_ACCESS, "url" => "https://technote.space/page-1483" ),
		"http_host" => array( "label" => "\$_SERVER[\"HTTP_HOST\"], which used to check referer", "type" => "string", "default" => COLLABORATIVE_FILTERING_HTTP_HOST, "url" => "https://technote.space/page-1486" ),
		"check_update" => array( "label" => "whether to check develop update", "type" => "bool", "default" => COLLABORATIVE_FILTERING_CHECK_UPDATE, "url" => "https://technote.space/page-1491" ),
		"test" => array( "label" => "whether to show test button", "type" => "bool", "default" => COLLABORATIVE_FILTERING_TEST, "url" => "https://technote.space/page-1495" ),
		"suppress_message" => array( "label" => "whether to suppress ajax message of access count", "type" => "bool", "default" => COLLABORATIVE_FILTERING_SUPPRESS_MESSAGE, "url" => "https://technote.space/page-1444" ),
	);

	private static $groups = array(
		"performance" => array(
			"sampling_rate",
			"minify_js",
			"minify_css",
			"utilize_wpp",
		),
		"security" => array(
			"check_data",
			"nonce_check",
			"check_referer",
			"suppress_message",
		),
		"general behavior" => array(
			"max_save_data_number",
			"jaccard_threshold",
			"jaccard_min_number",
			"post_types",
			"post_statuses",
		),
		"access behavior" => array(
			"exclude_loggedin_user",
			"update_cookie_expire",
			"user_expire",
		),
		"cron" => array(
			"calc_interval",
			"calc_timeout",
			"calculate_number",
			"calc_log",
			"clear_interval",
//			"clear_timeout",
			"data_expire",
			"clear_log",
		),
		"compatibility" => array(
			"front_admin_ajax",
			"consider_page_cache",
			"mod_rewrite_ajax_access",
			"url_scheme",
			"http_host",
		),
		"others" => array(
			"display_log_number",
			"check_update",
			"show_result",
			"test",
		)
	);

	protected function get_filter_prefix()
	{
		return 'cf_';
	}

	protected function apply_filters()
	{
		$args = func_get_args();
		$key = $args[0];
		$args[0] = $this->get_filter_prefix() . $key;

		if ( key_exists( $key, self::$settings ) ) {
			global $cf_option;
			$default = call_user_func_array( 'apply_filters', $args );
			$ret = $cf_option->get( $args[0] );
			if ( "" === $ret ) {
				$ret = $default;
			}
			switch ( self::$settings[$key]["type"] ) {
				case "bool":
					if ( is_bool( $ret ) ) {
						return $ret;
					}
					if ( "true" === $ret ) {
						return true;
					}
					if ( "false" === $ret ) {
						return false;
					}
					$ret = $default;
					break;
				case "int":
					$default = (int)$default;
					if ( is_numeric( $ret ) ) {
						$ret = (int)$ret;
						if ( $ret !== $default ) {
							if ( isset( self::$settings[$key]["min"] ) && $ret < (int)self::$settings[$key]["min"] ) {
								$ret = (int)self::$settings[$key]["min"];
							}
							if ( isset( self::$settings[$key]["max"] ) && $ret > (int)self::$settings[$key]["max"] ) {
								$ret = (int)self::$settings[$key]["max"];
							}
						} elseif ( isset( self::$settings[$key]["option"] ) ) {
							$default = isset( self::$settings[$key]["option_default"] ) ? (int)self::$settings[$key]["option_default"] : $default;
							$ret = (int)$cf_option->get( self::$settings[$key]["option"], $default );
						}
					} else {
						$ret = $default;
					}
					break;
				case "float":
					$default = (float)$default;
					if ( is_numeric( $ret ) ) {
						$ret = (float)$ret;
						if ( $ret !== $default ) {
							if ( isset( self::$settings[$key]["min"] ) && $ret < (float)self::$settings[$key]["min"] ) {
								$ret = (float)self::$settings[$key]["min"];
							}
							if ( isset( self::$settings[$key]["max"] ) && $ret > (float)self::$settings[$key]["max"] ) {
								$ret = (float)self::$settings[$key]["max"];
							}
						} elseif ( isset( self::$settings[$key]["option"] ) ) {
							$default = isset( self::$settings[$key]["option_default"] ) ? (float)self::$settings[$key]["option_default"] : $default;
							$ret = (float)$cf_option->get( self::$settings[$key]["option"], $default );
						}
					} else {
						$ret = $default;
					}
					break;
				default:
					break;
			}
			return $ret;
		}

		return call_user_func_array( 'apply_filters', $args );
	}

	protected function do_action()
	{
		$args = func_get_args();
		$args[0] = $this->get_filter_prefix() . $args[0];
		call_user_func_array( 'do_action', $args );
	}

	protected function get_settings( $tail_info = true )
	{
		return array_map( function ( $key, $value ) use ( $tail_info ) {

			return array(
				"label" => __( $key, COLLABORATIVE_FILTERING_TEXT_DOMAIN ),
				"settings" => array_map( function ( $key ) use ( $tail_info ) {
					$value = self::$settings[$key];

					global $cf_option;
					$value['key'] = $key;
					if ( !isset( $value['label'] ) ) {
						$value['label'] = $key;
					}
					$value['label'] = __( $value['label'], COLLABORATIVE_FILTERING_TEXT_DOMAIN );

					if ( $tail_info ) {
						$tail = '';
						$default = $this->get_expression( $value['default'], $value['type'] );
						if ( "" !== $default ) {
							$tail .= " [default = " . $default;
						}
						if ( isset( $value["min"] ) ) {
							$tail .= ", min = " . $this->get_expression( $value['min'], $value['type'] );
						}
						if ( isset( $value["max"] ) ) {
							$tail .= ", max = " . $this->get_expression( $value['max'], $value['type'] );
						}
						if ( "" !== $default ) {
							$tail .= "]";
						}
						if (isset($value["url"])) {
							$tail .= '<a href="' . esc_url($value['url']) . '" target="_blank" class="cf-detail-link">詳細</a>';
						}
						$value["label"] .= $tail;
					}

					$value['name'] = $this->get_filter_prefix() . $key;
					$value['db'] = $cf_option->get( $value['name'] );
					$value['placeholder'] = $this->get_expression( $value['default'], $value['type'] );
					$value["used"] = $this->get_expression( $this->apply_filters( $value["key"], $value["default"] ), $value['type'] );
					return $value;
				}, $value )
			);

		}, array_keys( self::$groups ), array_values( self::$groups ) );
	}

	protected function get_expression( $value, $type )
	{
		switch ( $type ) {
			case 'bool':
				return var_export( $value, true );
			case 'int':
				return $value;
			case 'float':
				return round( $value, 6 );
			default:
				return $value;
		}
	}

	protected function defined( $c )
	{
		if ( defined( $c ) ) {
			$const = @constant( $c );
			if ( $const )
				return true;
		}
		return false;
	}

	protected function definedv( $c, $default = NULL )
	{
		if ( defined( $c ) ) {
			$const = @constant( $c );
			return $const;
		}
		return $default;
	}

	protected function log( $message, $logfile = NULL, $error = false )
	{
		if ( !$this->defined( 'COLLABORATIVE_FILTERING_OUTPUT_LOG' ) )
			return false;
		if ( is_null( $logfile ) ) {
			$logfile = COLLABORATIVE_FILTERING_LOG_FILE;
		}

		$logdir = dirname( $logfile );
		if ( !file_exists( $logdir ) ) {
			@mkdir( $logdir, 0777, TRUE );
			@chmod( $logdir, 0777 );
		}

		@error_log( sprintf( "[%s] %s\n", date( 'Y-m-d H:i:s' ), $this->apply_filters( "log_message", is_string( $message ) ? $message : json_encode( $message ), $message ) ), 3, $logfile );
		if ( $error && $this->defined( 'COLLABORATIVE_FILTERING_SHOW_ERROR' ) )
			var_dump( nl2br( $message ) );
		return true;
	}

	protected function uuid()
	{
		//http://d.hatena.ne.jp/cakephper/20081218/1229544061
		$pid = getmypid();
		$node = $_SERVER['SERVER_ADDR'];
		list( $timeMid, $timeLow ) = explode( ' ', microtime() );
		return sprintf( "%08x%04x%04x%02x%02x%04x%08x", (int)$timeLow, (int)substr( $timeMid, 2 ) & 0xffff,
						mt_rand( 0, 0xfff ) | 0x4000, mt_rand( 0, 0x3f ) | 0x80, mt_rand( 0, 0xff ), $pid & 0xffff, $node );
	}

	protected function view( $name, $output = true, $args = array(), $lib = false )
	{
		extract( $args, EXTR_SKIP );
		ob_start();
		if ( $lib )
			@include( COLLABORATIVE_FILTERING_LIB_VIEWS_DIR . DIRECTORY_SEPARATOR . $name . ".php" );
		else @include( COLLABORATIVE_FILTERING_VIEWS_DIR . DIRECTORY_SEPARATOR . $name . ".php" );
		$view = ob_get_contents();
		ob_end_clean();
		if ( $output )
			echo $view;
		else return $view;
		return "";
	}

	protected function error( $error )
	{
		global $cf_controller;
		$cf_controller->add_error( $error );
	}

	protected function message( $message )
	{
		global $cf_controller;
		$cf_controller->add_message( $message );
	}

	protected function dir2path( $dir )
	{
		$dir = str_replace( COLLABORATIVE_FILTERING_ROOT_DIR, $this->get_plugin_url(), $dir );
		$dir = str_replace( DIRECTORY_SEPARATOR, "/", $dir );
		return $dir;
	}

	protected function path2dir( $path )
	{
		$path = str_replace( $this->get_plugin_url(), COLLABORATIVE_FILTERING_ROOT_DIR, $path );
		$path = str_replace( "/", DIRECTORY_SEPARATOR, $path );
		return $path;
	}

	protected function valid_post_types()
	{
		return array_filter( array_map( 'trim', explode( ',', $this->apply_filters( 'post_types', COLLABORATIVE_FILTERING_POST_TYPES ) ) ) );
	}

	protected function valid_post_statuses()
	{
		return array_filter( array_map( 'trim', explode( ',', $this->apply_filters( 'post_statuses', COLLABORATIVE_FILTERING_POST_STATUSES ) ) ) );
	}

	protected function get_server_key()
	{
		global $cf_option;
		$ret = $cf_option->get( 'server_key' );
		if ( $ret ) {
			return $ret;
		}
		$ret = wp_generate_password( 64, true, true );
		$cf_option->set( 'server_key', $ret );
		return $ret;
	}

	protected function init_server_key()
	{
		global $cf_option;
		$cf_option->set( 'server_key', '' );
	}

	protected function consider_page_cache()
	{
		return $this->apply_filters( 'consider_page_cache', COLLABORATIVE_FILTERING_CONSIDER_PAGE_CACHE );
	}

	protected function front_admin_ajax()
	{
		global $cf_test;
		return $cf_test->get_test_param( 'admin', $this->apply_filters( 'front_admin_ajax', COLLABORATIVE_FILTERING_FRONT_ADMIN_AJAX ) );
	}

	protected function get_url_scheme()
	{
		global $cf_test;
		return $cf_test->get_test_param( 'scheme', $this->apply_filters( 'url_scheme', COLLABORATIVE_FILTERING_URL_SCHEME ) );
	}

	protected function get_plugin_url( $dir = '', $lib = false, $scheme = null )
	{
		static $cache = array(), $cache_lib = array();

		if ( is_null( $scheme ) ) {
			$scheme = $this->get_url_scheme();
		}

		if ( $lib ) {
			$c = &$cache_lib;
			$d = COLLABORATIVE_FILTERING_LIB_COMMON_DIR;
		} else {
			$c = &$cache;
			$d = COLLABORATIVE_FILTERING_COMMON_DIR;
		}

		if ( !isset( $c[$scheme] ) ) {
			$c[$scheme] = set_url_scheme( plugins_url( '', $d ), $scheme );
		}

		$dir = ltrim( str_replace( DIRECTORY_SEPARATOR, '/', $dir ), '/' );
		return rtrim( $c[$scheme] . '/' . $dir, '/' );
	}

	protected function utilize_wpp()
	{
		return $this->apply_filters( 'utilize_wpp', COLLABORATIVE_FILTERING_UTILIZE_WPP ) && class_exists( 'WordpressPopularPosts' );
	}

	protected function mod_rewrite_ajax_access()
	{
		global $cf_test;
		return $cf_test->get_test_param( 'mod_rewrite', $this->apply_filters( 'mod_rewrite_ajax_access', COLLABORATIVE_FILTERING_MOD_REWRITE_AJAX_ACCESS ) );
	}

	protected function create_nonce( $action )
	{
		$i = wp_nonce_tick();

		return substr( wp_hash( $i . '|' . $action, 'nonce' ), -12, 10 );
	}

	protected function verify_nonce( $nonce, $action )
	{
		$nonce = (string)$nonce;

		if ( empty( $nonce ) ) {
			return false;
		}

		$i = wp_nonce_tick();

		$expected = substr( wp_hash( $i . '|' . $action, 'nonce' ), -12, 10 );
		if ( hash_equals( $expected, $nonce ) ) {
			return 1;
		}

		$expected = substr( wp_hash( ( $i - 1 ) . '|' . $action, 'nonce' ), -12, 10 );
		if ( hash_equals( $expected, $nonce ) ) {
			return 2;
		}

		return false;
	}

	protected function get_referer_host()
	{
		if ( !isset( $_SERVER['HTTP_REFERER'] ) ) {
			$referer = '';
		} else {
			$referer = $_SERVER['HTTP_REFERER'];
			$referer = parse_url( $referer );
			$referer = false === $referer ? '' : $referer['host'];
		}
		return $referer;
	}

	protected function get_http_host()
	{
		return $this->apply_filters( 'http_host', COLLABORATIVE_FILTERING_HTTP_HOST );
	}

	protected function check_referer( $test_referer = false, $test_host = false )
	{
		if ( false !== $test_referer ) {
			$referer = $test_referer;
		} else {
			$referer = $this->get_referer_host();
		}

		if ( false !== $test_host ) {
			$host = $test_host;
		} else {
			$host = $this->get_http_host();
		}
		$result = false !== stristr( $referer, $host );
		return array( 'result' => $result, 'host' => $host, 'referer' => $referer );
	}
}
